<?php
/**
 * StablePay 日志记录器
 *
 * @package StablePay_WooCommerce
 */

// 防止直接访问
if (!defined('ABSPATH')) {
    exit;
}

/**
 * StablePay_Logger 类
 *
 * 负责记录插件运行日志，支持不同日志级别
 */
class StablePay_Logger {
    /**
     * 日志级别常量
     */
    const LEVEL_ERROR = 'error';
    const LEVEL_WARNING = 'warning';
    const LEVEL_INFO = 'info';
    const LEVEL_DEBUG = 'debug';

    /**
     * WooCommerce Logger 实例
     *
     * @var WC_Logger
     */
    private $wc_logger;

    /**
     * 日志源标识
     *
     * @var string
     */
    private $source = 'stablepay-woocommerce';

    /**
     * 是否启用日志
     *
     * @var bool
     */
    private $enabled = false;

    /**
     * 日志级别
     *
     * @var string
     */
    private $level = self::LEVEL_INFO;

    /**
     * 构造函数
     *
     * @param bool $enabled 是否启用日志
     * @param string $level 日志级别
     */
    public function __construct($enabled = false, $level = self::LEVEL_INFO) {
        $this->enabled = $enabled;
        $this->level = $level;

        if (function_exists('wc_get_logger')) {
            $this->wc_logger = wc_get_logger();
        }
    }

    /**
     * 记录错误日志
     *
     * @param string $message 日志消息
     * @param array $context 上下文数据
     */
    public function error($message, $context = array()) {
        $this->log(self::LEVEL_ERROR, $message, $context);
    }

    /**
     * 记录警告日志
     *
     * @param string $message 日志消息
     * @param array $context 上下文数据
     */
    public function warning($message, $context = array()) {
        $this->log(self::LEVEL_WARNING, $message, $context);
    }

    /**
     * 记录信息日志
     *
     * @param string $message 日志消息
     * @param array $context 上下文数据
     */
    public function info($message, $context = array()) {
        $this->log(self::LEVEL_INFO, $message, $context);
    }

    /**
     * 记录调试日志
     *
     * @param string $message 日志消息
     * @param array $context 上下文数据
     */
    public function debug($message, $context = array()) {
        $this->log(self::LEVEL_DEBUG, $message, $context);
    }

    /**
     * 记录日志
     *
     * @param string $level 日志级别
     * @param string $message 日志消息
     * @param array $context 上下文数据
     */
    private function log($level, $message, $context = array()) {
        // 格式化日志消息
        $formatted_message = $this->format_message($message, $context);

        // 错误级别始终写入 error_log（便于调试）
        if ($level === self::LEVEL_ERROR) {
            error_log(sprintf('[StablePay][%s] %s', strtoupper($level), $formatted_message));
        }

        // 检查是否启用日志
        if (!$this->enabled) {
            return;
        }

        // 检查日志级别
        if (!$this->should_log($level)) {
            return;
        }

        // 使用 WooCommerce Logger
        if ($this->wc_logger) {
            $this->wc_logger->log($level, $formatted_message, array('source' => $this->source));
        } else {
            // 降级使用 error_log
            error_log(sprintf('[StablePay][%s] %s', strtoupper($level), $formatted_message));
        }
    }

    /**
     * 检查是否应该记录该级别的日志
     *
     * @param string $level 日志级别
     * @return bool
     */
    private function should_log($level) {
        $levels = array(
            self::LEVEL_ERROR => 4,
            self::LEVEL_WARNING => 3,
            self::LEVEL_INFO => 2,
            self::LEVEL_DEBUG => 1,
        );

        $current_level = isset($levels[$this->level]) ? $levels[$this->level] : 2;
        $log_level = isset($levels[$level]) ? $levels[$level] : 2;

        return $log_level >= $current_level;
    }

    /**
     * 格式化日志消息
     *
     * @param string $message 原始消息
     * @param array $context 上下文数据
     * @return string 格式化后的消息
     */
    private function format_message($message, $context = array()) {
        if (empty($context)) {
            return $message;
        }

        // 脱敏处理：隐藏敏感信息
        $sanitized_context = $this->sanitize_context($context);

        return $message . ' | Context: ' . json_encode($sanitized_context, JSON_UNESCAPED_UNICODE);
    }

    /**
     * 脱敏处理上下文数据
     *
     * @param array $context 原始上下文
     * @return array 脱敏后的上下文
     */
    private function sanitize_context($context) {
        $sensitive_keys = array('api_key', 'secret_key', 'webhook_secret', 'password', 'token');

        foreach ($context as $key => $value) {
            if (in_array(strtolower($key), $sensitive_keys) && is_string($value) && strlen($value) > 8) {
                // 只显示前4位和后4位
                $context[$key] = substr($value, 0, 4) . '****' . substr($value, -4);
            } elseif (is_array($value)) {
                $context[$key] = $this->sanitize_context($value);
            }
        }

        return $context;
    }
}
